/*
 * Decompiled with CFR 0.152.
 */
package com.graphhopper.reader.osm;

import com.carrotsearch.hppc.BitSet;
import com.carrotsearch.hppc.LongArrayList;
import com.graphhopper.coll.GHLongLongHashMap;
import com.graphhopper.reader.ReaderElement;
import com.graphhopper.reader.ReaderNode;
import com.graphhopper.reader.ReaderRelation;
import com.graphhopper.reader.ReaderWay;
import com.graphhopper.reader.dem.EdgeElevationSmoothingMovingAverage;
import com.graphhopper.reader.dem.EdgeElevationSmoothingRamer;
import com.graphhopper.reader.dem.EdgeSampling;
import com.graphhopper.reader.dem.ElevationProvider;
import com.graphhopper.reader.osm.OSMReaderUtility;
import com.graphhopper.reader.osm.OSMRestrictionConverter;
import com.graphhopper.reader.osm.OSMRestrictionException;
import com.graphhopper.reader.osm.RestrictionMembers;
import com.graphhopper.reader.osm.RestrictionTagParser;
import com.graphhopper.reader.osm.RestrictionTopology;
import com.graphhopper.reader.osm.RestrictionType;
import com.graphhopper.reader.osm.Triple;
import com.graphhopper.reader.osm.WaySegmentParser;
import com.graphhopper.reader.osm.WayToEdgesMap;
import com.graphhopper.routing.OSMReaderConfig;
import com.graphhopper.routing.ev.Country;
import com.graphhopper.routing.ev.EdgeIntAccess;
import com.graphhopper.routing.ev.State;
import com.graphhopper.routing.util.AreaIndex;
import com.graphhopper.routing.util.CustomArea;
import com.graphhopper.routing.util.FerrySpeedCalculator;
import com.graphhopper.routing.util.OSMParsers;
import com.graphhopper.routing.util.countryrules.CountryRule;
import com.graphhopper.routing.util.countryrules.CountryRuleFactory;
import com.graphhopper.routing.util.parsers.RestrictionSetter;
import com.graphhopper.search.KVStorage;
import com.graphhopper.storage.BaseGraph;
import com.graphhopper.storage.IntsRef;
import com.graphhopper.storage.NodeAccess;
import com.graphhopper.storage.TurnCostStorage;
import com.graphhopper.util.DistanceCalc;
import com.graphhopper.util.DistanceCalcEarth;
import com.graphhopper.util.EdgeIteratorState;
import com.graphhopper.util.FetchMode;
import com.graphhopper.util.GHUtility;
import com.graphhopper.util.Helper;
import com.graphhopper.util.PointList;
import com.graphhopper.util.RamerDouglasPeucker;
import com.graphhopper.util.StopWatch;
import com.graphhopper.util.shapes.GHPoint;
import com.graphhopper.util.shapes.GHPoint3D;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.LongToIntFunction;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OSMReader {
    private static final Logger LOGGER = LoggerFactory.getLogger(OSMReader.class);
    private static final Pattern WAY_NAME_PATTERN = Pattern.compile("; *");
    protected final OSMReaderConfig config;
    protected final BaseGraph baseGraph;
    private final EdgeIntAccess edgeIntAccess;
    private final NodeAccess nodeAccess;
    private final TurnCostStorage turnCostStorage;
    protected final OSMParsers osmParsers;
    private final DistanceCalc distCalc = DistanceCalcEarth.DIST_EARTH;
    private final RestrictionSetter restrictionSetter;
    protected ElevationProvider eleProvider = ElevationProvider.NOOP;
    private AreaIndex<CustomArea> areaIndex;
    private CountryRuleFactory countryRuleFactory = null;
    protected File osmFile;
    private final RamerDouglasPeucker simplifyAlgo = new RamerDouglasPeucker();
    private int bugCounter = 0;
    private final IntsRef tempRelFlags;
    protected Date osmDataDate;
    protected long zeroCounter = 0L;
    private GHLongLongHashMap osmWayIdToRelationFlagsMap = new GHLongLongHashMap(200, 0.5);
    private WayToEdgesMap restrictedWaysToEdgesMap = new WayToEdgesMap();
    private List<ReaderRelation> restrictionRelations = new ArrayList<ReaderRelation>();

    public OSMReader(BaseGraph baseGraph, OSMParsers osmParsers, OSMReaderConfig config) {
        this.baseGraph = baseGraph;
        this.edgeIntAccess = baseGraph.getEdgeAccess();
        this.config = config;
        this.nodeAccess = baseGraph.getNodeAccess();
        this.osmParsers = osmParsers;
        this.restrictionSetter = new RestrictionSetter(baseGraph, osmParsers.getRestrictionTagParsers().stream().map(RestrictionTagParser::getTurnRestrictionEnc).toList());
        this.simplifyAlgo.setMaxDistance(config.getMaxWayPointDistance());
        this.simplifyAlgo.setElevationMaxDistance(config.getElevationMaxWayPointDistance());
        this.turnCostStorage = baseGraph.getTurnCostStorage();
        this.tempRelFlags = osmParsers.createRelationFlags();
        if (this.tempRelFlags.length != 2) {
            throw new IllegalArgumentException("OSMReader cannot use relation flags with != 2 integers");
        }
    }

    public OSMReader setFile(File osmFile) {
        this.osmFile = osmFile;
        return this;
    }

    public OSMReader setAreaIndex(AreaIndex<CustomArea> areaIndex) {
        this.areaIndex = areaIndex;
        return this;
    }

    public OSMReader setElevationProvider(ElevationProvider eleProvider) {
        if (eleProvider == null) {
            throw new IllegalStateException("Use the NOOP elevation provider instead of null or don't call setElevationProvider");
        }
        if (!this.nodeAccess.is3D() && ElevationProvider.NOOP != eleProvider) {
            throw new IllegalStateException("Make sure you graph accepts 3D data");
        }
        this.eleProvider = eleProvider;
        return this;
    }

    public OSMReader setCountryRuleFactory(CountryRuleFactory countryRuleFactory) {
        this.countryRuleFactory = countryRuleFactory;
        return this;
    }

    public void readGraph() throws IOException {
        if (this.osmParsers == null) {
            throw new IllegalStateException("Tag parsers were not set.");
        }
        if (this.osmFile == null) {
            throw new IllegalStateException("No OSM file specified");
        }
        if (!this.osmFile.exists()) {
            throw new IllegalStateException("Your specified OSM file does not exist:" + this.osmFile.getAbsolutePath());
        }
        if (!this.baseGraph.isInitialized()) {
            throw new IllegalStateException("BaseGraph must be initialize before we can read OSM");
        }
        WaySegmentParser waySegmentParser = new WaySegmentParser.Builder(this.baseGraph.getNodeAccess(), this.baseGraph.getDirectory()).setElevationProvider(this::getElevation).setWayFilter(this::acceptWay).setSplitNodeFilter(this::isBarrierNode).setWayPreprocessor(this::preprocessWay).setRelationPreprocessor(this::preprocessRelations).setRelationProcessor(this::processRelation).setEdgeHandler(this::addEdge).setWorkerThreads(this.config.getWorkerThreads()).build();
        waySegmentParser.readOSM(this.osmFile);
        this.osmDataDate = waySegmentParser.getTimestamp();
        if (this.baseGraph.getNodes() == 0) {
            throw new RuntimeException("Graph after reading OSM must not be empty");
        }
        this.releaseEverythingExceptRestrictionData();
        this.addRestrictionsToGraph();
        this.releaseRestrictionData();
        LOGGER.info("Finished reading OSM file: {}, nodes: {}, edges: {}, zero distance edges: {}", new Object[]{this.osmFile.getAbsolutePath(), Helper.nf((long)this.baseGraph.getNodes()), Helper.nf((long)this.baseGraph.getEdges()), Helper.nf((long)this.zeroCounter)});
    }

    public Date getDataDate() {
        return this.osmDataDate;
    }

    protected double getElevation(ReaderNode node) {
        double ele = this.eleProvider.getEle(node);
        return Double.isNaN(ele) ? this.config.getDefaultElevation() : ele;
    }

    protected boolean acceptWay(ReaderWay way) {
        if (way.getNodes().size() < 2) {
            return false;
        }
        if (!way.hasTags()) {
            return false;
        }
        return this.osmParsers.acceptWay(way);
    }

    protected boolean isBarrierNode(ReaderNode node) {
        return node.hasTag("barrier", new String[0]) || node.hasTag("ford", new String[0]);
    }

    protected boolean isCalculateWayDistance(ReaderWay way) {
        return this.isFerry(way);
    }

    private boolean isFerry(ReaderWay way) {
        return FerrySpeedCalculator.isFerry(way);
    }

    protected void setArtificialWayTags(PointList pointList, ReaderWay way, double distance, List<Map<String, Object>> nodeTags) {
        CountryRule countryRule;
        List<Object> customAreas;
        way.setTag("node_tags", nodeTags);
        way.setTag("edge_distance", distance);
        way.setTag("point_list", pointList);
        way.removeTag("country");
        way.removeTag("country_rule");
        way.removeTag("custom_areas");
        if (this.areaIndex != null) {
            double middleLon;
            double middleLat;
            if (pointList.size() > 2) {
                middleLat = pointList.getLat(pointList.size() / 2);
                middleLon = pointList.getLon(pointList.size() / 2);
            } else {
                double firstLat = pointList.getLat(0);
                double firstLon = pointList.getLon(0);
                double lastLat = pointList.getLat(pointList.size() - 1);
                double lastLon = pointList.getLon(pointList.size() - 1);
                middleLat = (firstLat + lastLat) / 2.0;
                middleLon = (firstLon + lastLon) / 2.0;
            }
            customAreas = this.areaIndex.query(middleLat, middleLon);
        } else {
            customAreas = Collections.emptyList();
        }
        Country country = Country.MISSING;
        State state = State.MISSING;
        double countryArea = Double.POSITIVE_INFINITY;
        for (CustomArea customArea : customAreas) {
            String alpha2WithSubdivision;
            if (customArea.getProperties() == null || (alpha2WithSubdivision = (String)customArea.getProperties().get("ISO3166-2")) == null) continue;
            String[] strs = alpha2WithSubdivision.split("-");
            if (strs.length == 0 || strs.length > 2) {
                throw new IllegalStateException("Invalid alpha2: " + alpha2WithSubdivision);
            }
            Country c = Country.find(strs[0]);
            if (c == null) {
                throw new IllegalStateException("Unknown country: " + strs[0]);
            }
            if ((strs.length != 2 || state != State.MISSING && !(customArea.getArea() < countryArea)) && (strs.length != 1 || state != State.MISSING || !(customArea.getArea() < countryArea))) continue;
            country = c;
            state = State.find(alpha2WithSubdivision);
            countryArea = customArea.getArea();
        }
        way.setTag("country", (Object)country);
        way.setTag("country_state", (Object)state);
        if (this.countryRuleFactory != null && (countryRule = this.countryRuleFactory.getCountryRule(country)) != null) {
            way.setTag("country_rule", countryRule);
        }
        way.setTag("custom_areas", customAreas);
    }

    protected void addEdge(int fromIndex, int toIndex, PointList pointList, ReaderWay way, List<Map<String, Object>> nodeTags) {
        double distance;
        if (fromIndex < 0 || toIndex < 0) {
            throw new AssertionError((Object)("to or from index is invalid for this edge " + fromIndex + "->" + toIndex + ", points:" + String.valueOf(pointList)));
        }
        if (pointList.getDimension() != this.nodeAccess.getDimension()) {
            throw new AssertionError((Object)("Dimension does not match for pointList vs. nodeAccess " + pointList.getDimension() + " <-> " + this.nodeAccess.getDimension()));
        }
        if (pointList.size() != nodeTags.size()) {
            throw new AssertionError((Object)("there should be as many maps of node tags as there are points. node tags: " + nodeTags.size() + ", points: " + pointList.size()));
        }
        if (pointList.is3D()) {
            if (this.config.getLongEdgeSamplingDistance() < Double.MAX_VALUE) {
                pointList = EdgeSampling.sample(pointList, this.config.getLongEdgeSamplingDistance(), this.distCalc, this.eleProvider);
            }
            if (this.config.getElevationSmoothing().equals("ramer")) {
                EdgeElevationSmoothingRamer.smooth(pointList, this.config.getElevationSmoothingRamerMax());
            } else if (this.config.getElevationSmoothing().equals("moving_average")) {
                EdgeElevationSmoothingMovingAverage.smooth(pointList, this.config.getSmoothElevationAverageWindowSize());
            } else if (!this.config.getElevationSmoothing().isEmpty()) {
                throw new AssertionError((Object)("Unsupported elevation smoothing algorithm: '" + this.config.getElevationSmoothing() + "'"));
            }
        }
        if (this.config.getMaxWayPointDistance() > 0.0 && pointList.size() > 2) {
            this.simplifyAlgo.simplify(pointList);
        }
        if ((distance = this.distCalc.calcDistance(pointList)) < 0.001) {
            ++this.zeroCounter;
            distance = 0.001;
        }
        double maxDistance = 2147483.646;
        if (Double.isNaN(distance)) {
            LOGGER.warn("Bug in OSM or GraphHopper (" + this.bugCounter++ + "). Illegal tower node distance " + distance + " reset to 1m, osm way " + way.getId());
            distance = 1.0;
        }
        if (Double.isInfinite(distance) || distance > maxDistance) {
            LOGGER.warn("Bug in OSM or GraphHopper (" + this.bugCounter++ + "). Too big tower node distance " + distance + " reset to large value, osm way " + way.getId());
            distance = maxDistance;
        }
        if (this.bugCounter > 30) {
            throw new IllegalStateException("Too many bugs in OSM or GraphHopper encountered " + this.bugCounter);
        }
        this.setArtificialWayTags(pointList, way, distance, nodeTags);
        IntsRef relationFlags = this.getRelFlagsMap(way.getId());
        EdgeIteratorState edge = this.baseGraph.edge(fromIndex, toIndex).setDistance(distance);
        this.osmParsers.handleWayTags(edge.getEdge(), this.edgeIntAccess, way, relationFlags);
        Map<String, KVStorage.KValue> map = way.getTag("key_values", Collections.emptyMap());
        if (!map.isEmpty()) {
            edge.setKeyValues(map);
        }
        if (pointList.size() > 2) {
            this.checkCoordinates(fromIndex, (GHPoint)pointList.get(0));
            this.checkCoordinates(toIndex, (GHPoint)pointList.get(pointList.size() - 1));
            edge.setWayGeometry(pointList.shallowCopy(1, pointList.size() - 1, false));
        }
        this.checkDistance(edge);
        this.restrictedWaysToEdgesMap.putIfReserved(way.getId(), edge.getEdge());
    }

    private void checkCoordinates(int nodeIndex, GHPoint point) {
        double tolerance = 1.0E-6;
        if (Math.abs(this.nodeAccess.getLat(nodeIndex) - point.getLat()) > 1.0E-6 || Math.abs(this.nodeAccess.getLon(nodeIndex) - point.getLon()) > 1.0E-6) {
            throw new IllegalStateException("Suspicious coordinates for node " + nodeIndex + ": (" + this.nodeAccess.getLat(nodeIndex) + "," + this.nodeAccess.getLon(nodeIndex) + ") vs. (" + String.valueOf(point) + ")");
        }
    }

    private void checkDistance(EdgeIteratorState edge) {
        double tolerance = 1.0;
        double edgeDistance = edge.getDistance();
        double geometryDistance = this.distCalc.calcDistance(edge.fetchWayGeometry(FetchMode.ALL));
        if (Double.isInfinite(edgeDistance)) {
            throw new IllegalStateException("Infinite edge distance should never occur, as we are supposed to limit each distance to the maximum distance we can store, #435");
        }
        if (edgeDistance > 2000000.0) {
            LOGGER.warn("Very long edge detected: " + String.valueOf(edge) + " dist: " + edgeDistance);
        } else if (Math.abs(edgeDistance - geometryDistance) > 1.0) {
            throw new IllegalStateException("Suspicious distance for edge: " + String.valueOf(edge) + " " + edgeDistance + " vs. " + geometryDistance + ", difference: " + (edgeDistance - geometryDistance));
        }
    }

    protected void preprocessWay(ReaderWay way, WaySegmentParser.CoordinateSupplier coordinateSupplier, WaySegmentParser.NodeTagSupplier nodeTagSupplier) {
        long durationInSeconds;
        LinkedHashMap<String, KVStorage.KValue> map = new LinkedHashMap<String, KVStorage.KValue>();
        if (this.config.isParseWayNames()) {
            Map<String, Object> nodeTags;
            String nodeName;
            String bwdStr;
            String fwdStr;
            String string;
            Object name = "";
            if (!this.config.getPreferredLanguage().isEmpty()) {
                name = OSMReader.fixWayName(way.getTag("name:" + this.config.getPreferredLanguage()));
            }
            if (((String)name).isEmpty()) {
                name = OSMReader.fixWayName(way.getTag("name"));
            }
            if (!((String)name).isEmpty()) {
                map.put("street_name", new KVStorage.KValue(name));
            }
            if (!(string = OSMReader.fixWayName(way.getTag("ref"))).isEmpty()) {
                map.put("street_ref", new KVStorage.KValue(string));
            }
            if (way.hasTag("destination:ref", new String[0])) {
                map.put("street_destination_ref", new KVStorage.KValue(OSMReader.fixWayName(way.getTag("destination:ref"))));
            } else {
                fwdStr = OSMReader.fixWayName(way.getTag("destination:ref:forward"));
                bwdStr = OSMReader.fixWayName(way.getTag("destination:ref:backward"));
                if (!fwdStr.isEmpty() || !bwdStr.isEmpty()) {
                    map.put("street_destination_ref", new KVStorage.KValue(fwdStr.isEmpty() ? null : fwdStr, bwdStr.isEmpty() ? null : bwdStr));
                }
            }
            if (way.hasTag("destination", new String[0])) {
                map.put("street_destination", new KVStorage.KValue(OSMReader.fixWayName(way.getTag("destination"))));
            } else {
                fwdStr = OSMReader.fixWayName(way.getTag("destination:forward"));
                bwdStr = OSMReader.fixWayName(way.getTag("destination:backward"));
                if (!fwdStr.isEmpty() || !bwdStr.isEmpty()) {
                    map.put("street_destination", new KVStorage.KValue(fwdStr.isEmpty() ? null : fwdStr, bwdStr.isEmpty() ? null : bwdStr));
                }
            }
            LongArrayList nodes = way.getNodes();
            if (!nodes.isEmpty() && (way.hasTag("highway", (Object)"motorway") || way.hasTag("highway", (Object)"motorway_link")) && !(nodeName = (String)(nodeTags = nodeTagSupplier.getTags(nodes.get(0))).getOrDefault("name", "")).isEmpty() && "motorway_junction".equals(nodeTags.getOrDefault("highway", ""))) {
                map.put("motorway_junction", new KVStorage.KValue(nodeName));
            }
        }
        if (way.getTags().size() > 1) {
            for (Map.Entry entry : way.getTags().entrySet()) {
                if (!((String)entry.getKey()).endsWith(":conditional") || !(entry.getValue() instanceof String) || ((String)entry.getKey()).startsWith("maxspeed") || ((String)entry.getKey()).startsWith("maxweight")) continue;
                String value = KVStorage.cutString(((String)entry.getValue()).replace(" ", "").replace("bicycle", "bike"));
                String key = ((String)entry.getKey()).replace(':', '_').replace("bicycle", "bike");
                boolean fwd = key.contains("forward");
                boolean bwd = key.contains("backward");
                if (value.isEmpty()) continue;
                if (fwd == bwd) {
                    map.put(key, new KVStorage.KValue(value));
                    continue;
                }
                map.put(key, new KVStorage.KValue(fwd ? value : null, bwd ? value : null));
            }
        }
        way.setTag("key_values", map);
        if (!this.isCalculateWayDistance(way)) {
            return;
        }
        double distance = this.calcDistance(way, coordinateSupplier);
        if (Double.isNaN(distance)) {
            LOGGER.warn("Could not determine distance for OSM way: " + way.getId());
            return;
        }
        way.setTag("way_distance", distance);
        String durationTag = way.getTag("duration");
        if (durationTag == null) {
            if (this.isFerry(way) && distance > 500000.0) {
                GHUtility.OSM_WARNING_LOGGER.warn("Long ferry OSM way without duration tag: " + way.getId() + ", distance: " + Math.round(distance / 1000.0) + " km");
            }
            return;
        }
        try {
            durationInSeconds = OSMReaderUtility.parseDuration(durationTag);
        }
        catch (Exception e) {
            GHUtility.OSM_WARNING_LOGGER.warn("Could not parse duration tag '" + durationTag + "' in OSM way: " + way.getId());
            return;
        }
        double speedInKmPerHour = distance / 1000.0 / ((double)durationInSeconds / 60.0 / 60.0);
        if (speedInKmPerHour < 0.1) {
            GHUtility.OSM_WARNING_LOGGER.warn("Unrealistic low speed calculated from duration. Maybe the duration is too long, or it is applied to a way that only represents a part of the connection? OSM way: " + way.getId() + ". duration=" + durationTag + " (= " + Math.round((double)durationInSeconds / 60.0) + " minutes), distance=" + distance + " m");
            return;
        }
        way.setTag("speed_from_duration", speedInKmPerHour);
    }

    static String fixWayName(String str) {
        if (str == null) {
            return "";
        }
        return KVStorage.cutString(WAY_NAME_PATTERN.matcher(str).replaceAll(", "));
    }

    private double calcDistance(ReaderWay way, WaySegmentParser.CoordinateSupplier coordinateSupplier) {
        LongArrayList nodes = way.getNodes();
        GHPoint3D prevPoint = coordinateSupplier.getCoordinate(nodes.get(0));
        if (prevPoint == null) {
            return Double.NaN;
        }
        boolean is3D = !Double.isNaN(prevPoint.ele);
        double distance = 0.0;
        for (int i = 1; i < nodes.size(); ++i) {
            GHPoint3D point = coordinateSupplier.getCoordinate(nodes.get(i));
            if (point == null) {
                return Double.NaN;
            }
            if (Double.isNaN(point.ele) == is3D) {
                throw new IllegalStateException("There should be elevation data for either all points or no points at all. OSM way: " + way.getId());
            }
            distance += is3D ? this.distCalc.calcDist3D(prevPoint.lat, prevPoint.lon, prevPoint.ele, point.lat, point.lon, point.ele) : this.distCalc.calcDist(prevPoint.lat, prevPoint.lon, point.lat, point.lon);
            prevPoint = point;
        }
        return distance;
    }

    protected void preprocessRelations(ReaderRelation relation) {
        if (!relation.isMetaRelation() && relation.hasTag("type", (Object)"route")) {
            for (ReaderRelation.Member member : relation.getMembers()) {
                if (member.getType() != ReaderElement.Type.WAY) continue;
                IntsRef oldRelationFlags = this.getRelFlagsMap(member.getRef());
                IntsRef newRelationFlags = this.osmParsers.handleRelationTags(relation, oldRelationFlags);
                this.putRelFlagsMap(member.getRef(), newRelationFlags);
            }
        }
        Arrays.stream(OSMRestrictionConverter.getRestrictedWayIds(relation)).forEach(this.restrictedWaysToEdgesMap::reserve);
    }

    protected void processRelation(ReaderRelation relation, LongToIntFunction getIdForOSMNodeId) {
        if (this.turnCostStorage != null && OSMRestrictionConverter.isTurnRestriction(relation)) {
            long osmViaNode = OSMRestrictionConverter.getViaNodeIfViaNodeRestriction(relation);
            if (osmViaNode >= 0L) {
                int viaNode = getIdForOSMNodeId.applyAsInt(osmViaNode);
                if (viaNode >= 0) {
                    relation.setTag("graphhopper:via_node", viaNode);
                    this.restrictionRelations.add(relation);
                }
            } else {
                this.restrictionRelations.add(relation);
            }
        }
    }

    protected void addRestrictionsToGraph() {
        if (this.turnCostStorage == null) {
            return;
        }
        StopWatch sw = StopWatch.started();
        ArrayList<Triple<ReaderRelation, RestrictionTopology, RestrictionMembers>> restrictionRelationsWithTopology = new ArrayList<Triple<ReaderRelation, RestrictionTopology, RestrictionMembers>>(this.restrictionRelations.size());
        for (ReaderRelation restrictionRelation : this.restrictionRelations) {
            try {
                restrictionRelationsWithTopology.add(OSMRestrictionConverter.buildRestrictionTopologyForGraph(this.baseGraph, restrictionRelation, this.restrictedWaysToEdgesMap::getEdges));
            }
            catch (OSMRestrictionException e) {
                OSMReader.warnOfRestriction(restrictionRelation, e);
            }
        }
        ArrayList<RestrictionSetter.Restriction> restrictions = new ArrayList<RestrictionSetter.Restriction>();
        ArrayList<BitSet> encBits = new ArrayList<BitSet>();
        for (Triple triple : restrictionRelationsWithTopology) {
            try {
                BitSet bits = new BitSet((long)this.osmParsers.getRestrictionTagParsers().size());
                RestrictionType restrictionType = null;
                for (int i = 0; i < this.osmParsers.getRestrictionTagParsers().size(); ++i) {
                    RestrictionTagParser restrictionTagParser = this.osmParsers.getRestrictionTagParsers().get(i);
                    RestrictionTagParser.Result res = restrictionTagParser.parseRestrictionTags(((ReaderRelation)triple.first).getTags());
                    if (res == null) continue;
                    OSMRestrictionConverter.checkIfTopologyIsCompatibleWithRestriction((RestrictionTopology)triple.second, res.getRestriction());
                    if (restrictionType != null && res.getRestrictionType() != restrictionType) {
                        throw new OSMRestrictionException("has different restriction type for different vehicles.");
                    }
                    restrictionType = res.getRestrictionType();
                    bits.set((long)i);
                }
                if (bits.cardinality() <= 0L) continue;
                List<RestrictionSetter.Restriction> tmpRestrictions = OSMRestrictionConverter.buildRestrictionsForOSMRestriction(this.baseGraph, (RestrictionTopology)triple.second, restrictionType);
                restrictions.addAll(tmpRestrictions);
                tmpRestrictions.forEach(__ -> encBits.add(RestrictionSetter.copyEncBits(bits)));
            }
            catch (OSMRestrictionException e) {
                OSMReader.warnOfRestriction((ReaderRelation)triple.first, e);
            }
        }
        this.restrictionSetter.setRestrictions(restrictions, encBits);
        LOGGER.info("Finished adding turn restrictions. total turn cost entries: {}, took: {}", (Object)Helper.nf((long)this.baseGraph.getTurnCostStorage().getTurnCostsCount()), (Object)sw.stop().getTimeString());
    }

    private static void warnOfRestriction(ReaderRelation restrictionRelation, OSMRestrictionException e) {
        if (!e.isWithoutWarning()) {
            restrictionRelation.getTags().remove("graphhopper:via_node");
            List members = restrictionRelation.getMembers().stream().map(m -> m.getRole() + " " + m.getType().toString().toLowerCase() + " " + m.getRef()).collect(Collectors.toList());
            GHUtility.OSM_WARNING_LOGGER.warn("Restriction relation " + restrictionRelation.getId() + " " + e.getMessage() + ". tags: " + String.valueOf(restrictionRelation.getTags()) + ", members: " + String.valueOf(members) + ". Relation ignored.");
        }
    }

    protected void releaseEverythingExceptRestrictionData() {
        this.eleProvider.release();
        this.osmWayIdToRelationFlagsMap = null;
    }

    protected void releaseRestrictionData() {
        this.restrictedWaysToEdgesMap = null;
        this.restrictionRelations = null;
    }

    IntsRef getRelFlagsMap(long osmId) {
        long relFlagsAsLong = this.osmWayIdToRelationFlagsMap.get(osmId);
        this.tempRelFlags.ints[0] = (int)relFlagsAsLong;
        this.tempRelFlags.ints[1] = (int)(relFlagsAsLong >> 32);
        return this.tempRelFlags;
    }

    void putRelFlagsMap(long osmId, IntsRef relFlags) {
        long relFlagsAsLong = (long)relFlags.ints[1] << 32 | (long)relFlags.ints[0] & 0xFFFFFFFFL;
        this.osmWayIdToRelationFlagsMap.put(osmId, relFlagsAsLong);
    }

    public String toString() {
        return this.getClass().getSimpleName();
    }

    @FunctionalInterface
    public static interface Builder<ReaderClass> {
        public ReaderClass apply(BaseGraph var1, OSMParsers var2, OSMReaderConfig var3);
    }
}

